/*	con.c

GEMDOS console system

Originally written by JSL as noted below.

MODIFICATION HISTORY

	10 Mar 85	SCC	Added xauxout(), xprtout(), xauxin().
				(getch() can be used to perform function 0x07.)
				Updated rawconio() to spec (no 0xFE).
				Register optimization.

	11 Mar 85	SCC	Further register optimization.
				Added xconostat(), xprtostat(), xauxistat(), &
				xauxostat().
	12 Mar 85	SCC	Fixed xauxin().
	14 Mar 85	SCC	Extended path from BIOS character input through
				to returning the character to the user to be
				long.
				OOPS!  Repaired a '=' to a '==' in rawconio().
	19 Mar 85	SCC	Modified tests in conbrk() to just check low
				byte of character (ignoring scan code info).
	21 Mar 85	SCC	Modified conin()'s echo of long input character
				to int.
				Added definition of 'h' to conout parameters.
	25 Mar 85	SCC	Changed constat return to -1 (to spec).
				Modified xauxistat() to use constat() rather
				than bconstat() directly.
	 1 Apr 85	SCC	Added x7in().  getch() cannot be used directly
				for function 0x07.
	10 Apr 85	EWF	Installed circular buffer for typeahead.
	11 Apr 85	EWF	Modified ^R handling in cgets().
	12 Apr 85	EWF	Installed 'ring bell' on typeahead full.
			SCC	Installed EWF's changes of 10, 11 & 12 Apr 85.
	26 Apr 85	SCC	Modified ^X & ^C handling to flush BIOS buffer
				as well as BDOS buffer.
	29 Apr 85	SCC	Modified buffer flushing to flush just BDOS
				buffer, but to re-insert ^X into buffer.

NAMES

	JSL	Jason S. Loveman
	SCC	Steve C. Cavender
	EWF	Eric W. Fleischman
*/

/* console system for GEMDOS 3/6/85 JSL */

#include "fs.h"

/* *************************** typeahead buffer ************************* */
/*						*/	/* EWF  12 Apr 85 */
/* The following data structures are used for the typeahead buffer:	  */
/*									  */
long glbkbchar[3][KBBUFSZ];		/* The actual typeahead buffer	  */
					/* The 3 elements are prn,aux,con */
char kbchar[3];				/* size of typeahead buffer for   */
					/* each element			  */
long *insptr[3];			/* insertion ptr for each buffer  */
long *remptr[3];			/* removal ptr for each buffer	  */
/* ********************************************************************** */

int glbcolumn[3];

extern PD *run;

extern long trap13();					/* SCC  14 Mar 85 */

#define UBWORD(x) (((int) x) & 0x00ff)

#define   ctrlc  0x03
#define   ctrle  0x05
#define   ctrlq  0x11
#define   ctrlr  0x12
#define   ctrls  0x13
#define   ctrlu  0x15
#define   ctrlx  0x18

#define   cr      0x0d
#define   lf      0x0a
#define   tab     0x09
#define   rub     0x7f
#define   bs      0x08
#define   space   0x20

#define warmboot xterm(-32)

/* Function 0x0B */
/************************/
/* console input status */
/************************/

long
constat(h)
int h;
{
	return( kbchar[h] ? -1L : bconstat(h) );
}

long
xconstat()
{
	return(constat(run->p_uft[0]+3));
}


/* 0x10 */
/*************************/
/* console output status */
/*************************/

long
xconostat()
{
	return(bconostat(run->p_uft[1]+3));
}


/* 0x11 */
/*************************/
/* printer output status */
/*************************/

long
xprtostat()
{
	return(bconostat(run->p_uft[3]+3));
}

/* 0x12 */
/**************************/
/* auxillary input status */
/**************************/

long
xauxistat()
{
	return(constat(run->p_uft[2]+3));
}


/* 0x13 */
/***************************/
/* auxillary output status */
/***************************/

long
xauxostat()
{
	return(bconostat(run->p_uft[2]+3));
}


/********************/
/* check for ctrl/s */
/* used internally  */
/********************/
conbrk(h)
int h;
{
    register long ch;
    register int stop, c;

    stop = 0;
    if ( bconstat(h) )
	do
	{
		c = (ch = bconin(h)) & 0xFF;
		if ( c == ctrlc )
		{
			buflush(h);	/* flush BDOS & BIOS buffers */
			return(warmboot);
		}

		if ( c == ctrls )
			stop = 1;
		else if ( c == ctrlq )
			stop = 0;
		else if ( c == ctrlx )
		{
			buflush(h);
			*insptr[h]++ = ch;
			kbchar[h]++;
		}
		else
		{
			if (kbchar[h] < KBBUFSZ)
			{
				*insptr[h]++ = ch;
				kbchar[h]++;
			}
			else
				bconout(h, 7);
		}
	} while (stop);
}

buflush(h)
{
	/* flush BDOS type-ahead buffer */

	kbchar[h] = 0;
	remptr[h] = insptr[h] = &(glbkbchar[h][0]);
}

/******************/
/* console output */
/* used internally*/
/******************/

conout(h,ch)
int h, ch;
{
    conbrk(h);			/* check for control-s break */
    bconout(h,ch);		/* output character to console */
    if (ch >= ' ') glbcolumn[h]++;	/* keep track of screen column */
    else if (ch == cr) glbcolumn[h] = 0;
    else if (ch == bs) glbcolumn[h]--;
}

/* 0x02 */
/*************************************/
/* console output with tab expansion */
/*************************************/

xtabout(ch)
int ch;
{
	tabout(run->p_uft[1]+3,ch);
}

tabout(h,ch)
int h;
int ch;		/* character to output to console	*/
{
    if (ch == tab) do
	conout(h,' ');
    while (glbcolumn[h] & 7);
    else conout(h,ch);
}

/*******************************/
/* console output with tab and */
/* control character expansion */
/*******************************/

cookdout(h,ch)
int h;
int ch;		/* character to output to console	*/
{
    if (ch == tab) tabout(h,ch); /* if tab, expand it	*/
    else
    {
	if ( ch < ' ' )
	{
            conout( h,'^' );
	    ch |= 0x40;
	}
    conout(h,ch);			/* output the character */
    }
}

/* 0x04 */
/********************/
/* auxillary output */
/********************/

xauxout(ch)
int ch;
{
	bconout(run->p_uft[2]+3,ch);
}

/* 0x05 */
/******************/
/* printer output */
/******************/

xprtout(ch)
int ch;
{
	bconout(run->p_uft[3]+3,ch);
}


long
getch(h)
int h;
{
	long temp;

	if (kbchar[h])
	{
		temp = *remptr[h]++;
		if (!--kbchar[h])
			remptr[h] = insptr[h] = &(glbkbchar[h][0]);
		return(temp);
	}

	return(bconin(h));
}

/* 0x07 */
/*************************************/
/* direct console input without echo */
/*************************************/

long
x7in()
{
	return(getch(run->p_uft[0]+3));
}


long
conin(h)		/* BDOS console input function */
int h;
{
    long ch;

    conout( h,(int)(ch = getch(h)) );
    return(ch);
}

/* 0x01 */
/*****************/
/* console input */
/*****************/

long
xconin()
{
	return(conin(run->p_uft[0]+3));
}


/* 0x08 */
/******************************/
/* console input without echo */
/******************************/

long
x8in()
{
	register int h;
	register long ch;

	h = run->p_uft[0]+3;
	ch = getch(h);
	conbrk(h);
	return(ch);
}

/* 0x03 */
/*******************/
/* auxillary input */
/*******************/

long
xauxin()
{
	return(bconin(run->p_uft[2]+3));
}


/* 0x06 */
/******************
* raw console i/o *
******************/

long
rawconio(parm)
int parm;
{
	int i;

	if (parm == 0xFF)
	{
		i = run->p_uft[0]+3;
		return(constat(i) ? getch(i) : 0L);
	}
	bconout(run->p_uft[1]+3, parm);
}


/* 0x09 */
/*******************************************************/
/* print line up to delimiter(null) with tab expansion */
/*******************************************************/

xprt_line(p)
char *p;
{
	prt_line(run->p_uft[1]+3,p);
}

prt_line(h,p)
int h;
char *p;
{
    while( *p ) tabout( h, *p++ );
}


/**********************************************/
/* read line with editing and bounds checking */
/**********************************************/

/* Two subroutines first */

newline(h,startcol)
int startcol,h;
{
    conout(h,cr);			/* go to new line */
    conout(h,lf);
    while(startcol)
    {
	conout(h,' ');
	startcol -= 1;		/* start output at starting column */
    }
}


backsp(h,cbuf,retlen, col) /* backspace one character position */
int h;
int retlen;
char *cbuf;
int col;			/* starting console column	*/
{
    register char	ch;		/* current character		*/
    register int	i;
    register char	*p;		/* character pointer		*/

    if (retlen) --retlen;
				/* if buffer non-empty, decrease it by 1 */
    i = retlen;
    p = cbuf;
    while (i--)			/* calculate column position 	*/
    {				/*  across entire char buffer	*/
	ch = *p++;		/* get next char		*/
	if ( ch == tab )
	{
	    col += 8;
	    col &= ~7;		/* for tab, go to multiple of 8 */
	}
	else if ( ch < ' ' ) col += 2;
				/* control chars put out 2 printable chars */
	else col += 1;
    }
    while (glbcolumn[h] > col)
    {
	conout(h,bs);		/* backspace until we get to proper column */
	conout(h,' ');
	conout(h,bs);
    }
    return(retlen);
}

readline(p) /* function 10 */
char *p; /* max length, return length, buffer space */
{
	p[1] = cgets(run->p_uft[0]+3,(((int) p[0]) & 0x0ff),&p[2]);
}

cgets(h,maxlen,buf)
int h;	/* h is special handle denoting device number */
int maxlen;
char *buf;
{
	char ch;
	int i,stcol,retlen;

	stcol = glbcolumn[h];		/* set up starting column */
	for (retlen = 0; retlen < maxlen; )
	{
		switch(ch = getch(h))
		{
			case cr:
			case lf: conout(h,cr); goto getout;
			case bs:
			case rub:
				retlen = backsp(h,buf,retlen,stcol);
				break;
			case ctrlc: warmboot;
			case ctrlx:
				do retlen = backsp(h,buf,retlen,stcol);
				while (retlen);
				break;
			case ctrlu:
				conout(h,'#'); 
				newline(h,stcol);
				retlen = 0;
				break;
			case ctrlr:
				conout(h,'#');
				newline(h,stcol);
				for (i=0; i < retlen; i++)
					cookdout(h,buf[i]);
				break;
			default:
				cookdout(h,buf[retlen++] = ch);
		}
	}
getout:	return(retlen);
}
                                                  