/*  fs.c - GEMDOS file system procedures  */

/*
**
** GEMDOS file system
** ------------------
** Originally written by JSL as noted below.
**
** MODIFICATION HISTORY
**
**	13 Mar 85	SCC	Changed xgetfree() to return values into longs.
**				(As per spec).
**
**				Changed handling of getbpb() return to allow 
**				flagging of bad getbpb().
**
**	14 Mar 85	JSL	Modified getrec() to clear b_dirty flag.
**
**				Modified ixcreat() to force flush
**				(for the sake of xmkdir() ).
**
**	22 Mar 85	SCC	Modified xsetdrv() to return drive map (to 
**				bring up to spec functionality, specifically 
**				at request of Steve Schmitt (DR Logo) ).
**
**				Extended good/bad returns based on ckdrv() 
**				results.
**
**	26 Mar 85	JSL	Modified xsfirst() to be a front end to 
**				ixsfirst().  ixsfirst() is used internally by 
**				xexec() to force a disk access prior to 
**				xpgmld() so that media change info can be 
**				updated sooner.
**
**				Modified ixlseek() because of bug when pointer 
**				is at end of cluster prior to forward seek.
**
**			SCC	Made above actual changes from JSL's notes.
**
**	27 Mar 85	SCC	Modifed dcrack() and findit() to terminate with
**				indication of no file found if bad return from 
**				ckdrv().
**
**	28 Mar 85	SCC	Modified xchdir() to look for NEGATIVE return 
**				from ckdrv().
**
**	29 Mar 85	JSL	Fixed xrmdir() bug of "ghost" subdirectories.
**				Fixed problem creating files in unaccessed 
**				subdir.
**
**	 4 Apr 85	SCC	Modified several functions to improve 
**				readability.
**
**				Removed a number of old 'SCC  ?? ??? 85' 
**				modification marks, and began adding formfeeds 
**				between routines and routine headers.
**
**				Modified dup() to return long value and distinct
**				error codes.
**
**				Modified xforce() and ixforce() to return long 
**				value and distinct error codes, and to range 
**				check 'h'.
**
**				Now includes gemerror.h.
**
**				Modified xclose() and ixclose() to return long 
**				value and distinct error codes.
**
**	 5 Apr 85	SCC	Modified ixforce() to range check 'std', and 
**				moved it after xforce().
**
**	 8 Apr 85	SCC	Added declaration of 'drv' parameter to log().
**
**				Added 'long' type to declaration of flush() 
**				and error returns.
**
**				Added 'long' type to declaration of usrio().
**
**				Added 'int' type to declaration of getpath() 
**				and 'flg' parameter.
**
**				Added 'long' type to declaration of getcl() 
**				and error returns.
**
**				Added 'long' type to declaration of ckdrv() 
**				and error returns.
**
**				Added 'long' type to declaration of opnfil().
**
**				Added 'long' type to declaration of makopn().
**
**	 9 Apr 85	SCC	Added 'int' type to declaration of log2().
**
**				Added 'int' type to declaration of divmod().
**
**	10 Apr 85	SCC	Removed 'long' return from flush(), reversing
**				modification of 8 Apr 85, since errors from 
**				the BIOS are handled by longjmp()ing back to 
**				top of dispatcher.
**
**				Removed 'long' return from usrio(), reversing
**				modification of 8 Apr 85, since errors from 
**				the BIOS are handled by longjmp()ing back to 
**				top of dispatcher.
**
**				Changed definition of rwerr from 'int' to 'long'
**
**				Modified ixlseek() to return ERANGE and EINTRN.
**
**				Modified xwrite() to check validity of handle.
**
**				Modified ixclose() to check error returns from 
**				ixlseek().
**
**				Modified ixclose() to return EINTRN where JSL 
**				had marked 'some kind of internal error', and 
**				to return E_OK at end.
**
**				Modified getcl() to return EINTRN.
**
**				Modified nextcl() to return EINTRN and E_OK.
**
**				Added rc to scan() to check error returns from 
**				ixlseek().
**
**				Modified xsnext() to return E_OK and ENMFIL.
**
**				Modified xsfirst() to return E_OK and ENMFIL.
**
**	11 Apr 85	SCC	Modified opnfil() to return ENHNDL.
**
**				Modified ixcreat() to return EPTHNF, EACCDN.
**
**				Modified xrename() to return errors.
**
**				Added xfreset() and ixfreset().
**
**	12 Apr 85	SCC	Removed spurious ',0' from parameters passed 
**				to xclose() by xfreset().
**
**	14 Apr 85	SCC	Backed out modification of 11 Apr 85 that added
**				xfreset() and ixfreset().  They were not the 
**				solution to the problem they were aimed at 
**				fixing.  See corresponding note in the CLI 
**				about the ^C problem.
**
**				Modified ixdel() to not delete an open file.
**
**				Backed out modifications to getcl() (it is now 
**				'int' and returns -1).
**
**				Backed out modifications to nextcl() (it now 
**				returns -1).
**
**				Backed out modifications to clfix() (it us 
**				untyped).
**
**	16 Apr 85	SCC	Modified ixdel() to close the file if it is 
**				open, and then go ahead and delete it.  This 
**				fix was in response to the fact that AS68 
**				routinely deletes its files without closing 
**				them.
**
**	29 Apr 85	SCC	Modified xrename() to check for existence of 
**				new file name before attempting to rename old 
**				file.
**
**	 1 May 85	SCC	Did slight code optimization on ixsfirst().
**
**	 6 May 85	SCC	Modified ixsfirst() to report EFILNF on error 
**				return from findit().
**
**				Modified ixcreat() to return EPTHNF for null 
**				file name.
**
**	 7 May 85	SCC	Modified xchdir() to return EPTHNF on failure of
**				findit().
**
**	 8 May 85	SCC	Modified xchdir() to not change path on failure
**				of findit().
**
**	 9 May 85	SCC	Modified xrename() to return EACCDN if 
**				destination filename already exists.
**
**	13 May 85	SCC	Modified xchdir() to call ucase() before path 
**				string is used.
**
**				Modified findit() to call ucase() before name 
**				string is used.
**
**				Modified builds() to truncate pre-'.' portion 
**				of file name to 8 characters.
**
**				Modified xmkdir() to use ixcreat() instead of 
**				xcreat().
**
**				Modified xcreat() to prevent external caller 
**				from creating a subdirectory.
**
**	15 May 85	SCC	Modified xgetfree() to be 0=default, 1=A:, etc.
**
**	16 May 85	SCC	Modified builds() to terminate post-'.' 
**				portion of file name upon scanning a '.' as 
**				well as the other characters it was checking.
**
**        26 Jun 85       LTG     Fixed bug in xrename to return err from open.
**
**        27 Jun 85       LTG     Added "o_mod" parm to OFD data structure.
**
**				Modified xopen() to return EACCDN err if try 
**				to open file with read only mode in read/write 
**				or write mode.
** 
**				Mod to xread() to return EACCDN err if try to
**				read file opened as write only.
** 
**				Modifiedixwrite()toreturnEACCDNerriftry
**				to write file opend as read only.
**
**				Modified getdmd() to return NULPTR if MGET 
**				failed.
**
**				Modified log() to return a long indicating 
**				ENSMEM if getdmd() failed.
**
**				Modified makofd() to return NULPTR if MGET 
**				fails.
**
**				Modified makdnd() to return NULPTR if MGET 
**				fails.
** 
**				Modified makopn() to return ENSMEM if MGET 
**				fails.
**
**				Modified scan() to return NULPTR if makofd()
**				or makdnd() fails.
**
**				Modified xcreate() to return (ENSMEM) if 
**				makofd() fails.
**
**				Modified xmkdir() to return ENSMEM if makofd()
**				or makdnd() fails.
**
**				Modified rmdir() to return ENSMEM if makofd() 
**				fails.
**
**        19 Jul 85       LTG	Modified scan() to make sure a file has not be 
**				deleted before creating a DND for it.  This 
**				fixes the bug that prevented some directories 
**				from being removed.
**
**	19 Jul 85	SCC	Modified scan() to prevent creation of a new 
**				DND for a subdirectory that already has one.
**
**				Added routine uc() to upper-case a single 
**				character and removed up_string().
**
**				Modified dcrack() to use uc().
**
**				Modified xcmps() to use uc().
**
**				Modified xchdir() to use uc().
**
**				Modified findit() to not use up_string().
**
**				Modified builds() to use uc().
**
**				Modified match() to use uc().
**
**	22 Jul 85	LTG	Modified ixcreate() to pass mode parameter to 
**				opnfl(), 0 for RO, 2 for RW.
**
**			SCC	Modified scan().  Modification of 19 Jul 85 
**				was not correct.
**
**				Modified ixcreat().  Modification of 22 Jul 85 
**				did not check for R/O status correctly.
**
**	23 Jul 85	SCC	Modified scan().  Still in pursuit of 
**				corrections to mods made on 22 Jul 85.
**
** 	23 Jul 85	LTG	Modified builds() to chk for SLASH when namd is
**				8 chs long.  This fixes the bug with 8 char 
**				directory nms.
**
**	24 Jul 85	SCC	Modified scan().  (Snide comment about still not
**				having fixed the DND problem last referred to on
**				23 Jul 85.)
**
**	25 Jul 85	SCC	Modified xunlink().  It now reports correctly 
**				EACCDN if the file being removed is read-only.
**
** 	26 Jul 85	LTG	Modified xrename(). Fixed call to getofd() to 
**				pass an int instead of a long.
**
**			SCC	Modified ixread().  Caller could pass in long
**				negative length, which was causing problems.
**
**	29 Jul 85	SCC	Modified ixcreat() to disallow creation of an 
**				entry beginning with '.', specifically to fix 
**				'MD .'.
**
**				Modified xrmdir() to disallow 'RD .' or 
**				'RD ..'.
**
**	31 Jul 85	LTG	Modified ixlseek(). Now chks to see if at front of
** 				file before bumping cluster num when on cluster 
** 				boundry.
**
**	 6 Aug 85	LTG	Modified xchdir().  No longer removes drive specification
**				from path name before sending it to findit().
**
**	 7 Aug 85	LTG	Modified makdmd() to deallocate memory of just allocated
**				data structures if it runs out of mem before it's done.
**
**  mod       who date		what
**  --------- --- ---------	----
**  M00.14.01 ktb 02 Aug 86	Fix to xrw enabling proper deblocking of 
**				requests where cluster sizes are not 2.
**
** NOTES
**	SCC	 4 Apr 85	Note about bcb management in getrec().
**
** NAMES
**
**	JSL	Jason S. Loveman
**	SCC	Steven C. Cavender
**	LTG     Louis T. Garavaglia
**	KTB	Karl T. Braun (kral)
**
** ---------------------------------------------------------------------
** DOS 2.0 Media compatible file system
** by Jason S. Loveman
**			15 December, 1984
** 		15:05 - 21 January, 1985 >> changed file number stuff
** 		14:35 - 04 February, 1985 >> fixup sft owners, usecounts
** Last update: 16:35 - 21 February, 1985 >> add multi-sector I/O support
** 
** Notes:
**	1. Cluster size must be < 32K bytes (strictly less)
**	2. Cluster size, record size must all be powers of two
**	3. With current 12-bit FAT implemented,
**		maximum media size is approx. 4000 clusters
**	4. 16-bit FAT entries not implemented yet
**	5. swap routines must be defined accordingly for media
**		compatibility with 8086 (high-low) processors
**
*/

#include "fs.h"
#include "gemerror.h"

/*
**  external declarations
*/

extern PD *run;
extern BCB *bufl[2];
extern long errbuf[3];
extern char *xmgetblk();
extern long trap13();


/*
**  global declarations
*/

	/*
	**  drvsel - mask of drives selected since power up 
	*/

	int drvsel; 

	/*
	**  rwerr -  hard error number currently in progress 
	*/

	long rwerr; 

	/*
	**  errdrv -  drive on which error occurred 
	*/

	int errdrv; 

	/*
	**  logmsk -
	**	log values of:
	**        1, 2, 4, 8, 16, 32, 64, 128  256, 512, 1024, 2048 ... 
	*/

	int logmsk[] = {  0, 1, 3, 7, 15, 31, 63, 127, 255, 511, 1023, 2047,
		 4095, 8191, 16383, 32767 } ;

	/*
	**  negone -
	*/

	long negone = { -1L } ;

	/*
	**  dots -, dots2  -
	*/

	char dots[22] =  { ".          " } ;
	char dots2[22] = { "..         " } ;

	/*
	**  time -, date -
	*/

	int time,date;

	/*
	**  sft -
	*/

	FTAB sft[OPNFILES];

	/*
	**  dirtbl -
	** 	point these at DNDs when needed	
	**  diruse -
	**	use count 
	*/

	DND *dirtbl[NCURDIR];		
	char diruse[NCURDIR];		

	/*
	**  drvtbl -
	*/

	DMD *drvtbl[16];


/*
**  forward declarations
*/

long xread(), xwrite(), xlseek(), xrw(), ixread(), ixwrite(), ixlseek(), 
	ixforce();
long ixsfirst(), ixcreat(), ixopen(), ixdel(), ixclose(), makopn(), log();




/*
**  uc() - 
**	utility routine to return the upper-case of character passed in
**
**	Last modified	19 Jul 85	SCC
*/

uc(c)
char c;
{
	return((c >= 'a') && (c <= 'z') ? c & 0x5F : c);
}

/*
**  getdmd -
**	allocate storage for and initialize a DMD
*/

DMD	*getdmd(drv)
	int drv;
{
	DMD *dm;

	if (!(drvtbl[drv] = dm = MGET(DMD))) return (NULPTR);
	if (!(dm->m_dtl = MGET(DND))) goto fredm;
	if (!(dm->m_dtl->d_ofd = MGET(OFD))) goto fredtl;
	if (!(dm->m_fatofd = MGET(OFD))) goto freofd;

	return(dm);

freofd:	xmfreblk (dm->m_dtl->d_ofd);
fredtl:	xmfreblk (dm->m_dtl);
fredm:	xmfreblk (dm);
	return (NULPTR);
}


/*
**  syshnd -
*/

syshnd(h)
	int h;
{
	if (h >= NUMSTD)
		return(h-NUMSTD);

	if ((h = run->p_uft[h]) > 0 )
		return(h-NUMSTD);

	return(h);
}


/*
**  getofd -
*/

OFD	*getofd(h)
	int h;
{
	return(sft[syshnd(h)].f_ofd);
}


/*
**  ixdirdup -
*/

ixdirdup(h,dn,p)
	PD *p;
	int h;			/* file handle				*/
	int dn;			/* directory number			*/
{
	p->p_curdir[h] = dn;
	diruse[dn]++;
}


/*
**  dup -
**	duplicate a file handle.
**
**	Function 0x45	f_dup
**
**	Error returns
**		EIHNDL
**		ENHNDL
**
**	Last modified	SCC	 5 Apr 85
*/

long	dup(h)	
	int h;			/*+ h must be a standard handle (checked) */
{
	register int i;

	if ((h<0) || (h >= NUMSTD))
		return(EIHNDL);		/* only dup standard */

	for (i = 0; i < OPNFILES; i++)	/* find the first free handle */
		if (!sft[i].f_own)
			break;

	if (i == OPNFILES)
		return(ENHNDL);		/* no free handles */

	sft[i].f_own = run;

	if ((h = run->p_uft[h]) > 0)
		sft[i].f_ofd = sft[h-NUMSTD].f_ofd;
	else
		sft[i].f_ofd = (long) h;

	sft[i].f_use = 1;

	return(i+NUMSTD);
}


/*
**  xforce -
**
**	Function 0x46	f_force
**
**	Error returns
**		EIHNDL
**
**	Last modified	SCC	5 Apr 85
*/

long	xforce(std,h)
	int std,h;
{
	return(ixforce(std,h,run));
}

/*
**  ixforce -
*/

long	ixforce(std,h,p)	
	PD *p;			
	int std;		/* std must	be a standard handle 	*/
	int h;			/* h   must NOT be a standard handle	*/
{
	long fh;

	if ((std < 0) || (std >= NUMSTD))
		return(EIHNDL);
	
	if (h < 0)
		p->p_uft[std] = h;
	else
	{
		if (h < NUMSTD)
			return(EIHNDL);

		if ((fh = sft[h-NUMSTD].f_ofd) < 0L)
			p->p_uft[std] = fh;
		else
		{
			p->p_uft[std] = h;
			sft[h-NUMSTD].f_use++;
		}
	}
	return(E_OK);
}


/*
**  log2 -
**	return log base 2 of n
*/

int	log2(n) 
	int n;
{
	int i;

	for (i = 0; n ; i++)
		n >>= 1;

	return(i-1);
}


/*
**  log -
**	log in media 'b' on drive 'drv'.
**
**	Last modified	SCC	8 Apr 85
*/

long	log(b,drv) 
	BPB *b;
	int drv;
{
	OFD *fd,*fo,*f;
	DND *d;
	DMD *dm;
	int rsiz,cs,n,fs,ncl,fcl;

	rsiz = b->recsiz;
	cs = b->clsiz;
	n = b->rdlen;
	fs = b->fsiz;
	if (!(dm = getdmd(drv))) return (ENSMEM);
	d = dm->m_dtl;
	dm->m_fsiz = fs;			/* fat size */
	f = d->d_ofd;				/* root dir file */
	dm->m_drvnum = drv;
	f->o_dmd = dm;
	d->d_drv = dm;
	d->d_name[0] = 0;
	dm->m_16 = b->b_flags & 1;
	dm->m_clsiz = cs;
	dm->m_clsizb = b->clsizb;
	dm->m_recsiz = rsiz;
	dm->m_numcl = b->numcl;
	dm->m_clrlog = log2(cs);
	dm->m_clrm = logmsk[dm->m_clrlog];
	dm->m_rblog = log2(rsiz);
	dm->m_rbm = logmsk[dm->m_rblog];
	dm->m_clblog = log2(dm->m_clsizb);
	f->o_fileln = n * rsiz;

	/* number of "clusters" for root dir */
	ncl = (n + cs - 1)/cs;
	d->d_strtcl = f->o_strtcl = -1 - ncl;
	fcl = (fs + cs - 1)/cs;
	fo = dm->m_fatofd;			/* fat "file" */
	fo->o_strtcl = d->d_strtcl - fcl;
	fo->o_dmd = dm;
	dm->m_recoff[0] = b->fatrec - (fo->o_strtcl * cs);
	dm->m_recoff[1] = (b->fatrec + fs) - (d->d_strtcl * cs);

	/* 2 is first data cluster */
	dm->m_recoff[2] = b->datrec - (2 * cs);
	fo->o_bytnum = 3;
	fo->o_curbyt = 3;
	fo->o_fileln = fs * rsiz;

	return (0L);
}


/*
**  cl2rec -
*/

cl2rec(cl,dm)
	int cl;
	DMD *dm;
{
	return(cl * dm->m_clsiz);
}


/*
**  xfr2usr -
*/

xfr2usr(n,s,d)
	int n;
	char *s,*d;
{
	while (n--)
		*d++ = *s++;
}


/*
**  uxr2xfr -
*/

usr2xfr(n,d,s)
	int n;
	char *s,*d;
{
	while (n--)
		*d++ = *s++;
}


/*
**  xmovs -
*/

xmovs(n,s,d)
	int n;
	char *s,*d;
{
	while (n--)
		*d++ = *s++;
}

/*  
**  xcmps() - 
**	utility routine to compare two 11-character strings
**
**	Last modified	19 Jul 85	SCC
*/

int	xcmps(s,d)
	char *s,*d;
{
	int i;

	for (i = 0; i < 11; i++)
		if (uc(*s++) != uc(*d++))
			return(0);
	return(1);
}

/*
**  xclose -
**
**	Function 0x3E	f_close
**
**	Error returns
**		EIHNDL
**		ixclose()
**
**	Last modified	SCC	10 Apr 85
**
**	SCC	I have added 'rc' to allow return of status from ixclose().  I 
**		do not yet know whether it is appropriate to perform the 
**		operations inside the 'if' statement following the invocation 
**		of ixclose(), but I am leaving the flow of control intact.
*/

long	xclose(h)
	int h;
{
	int h0;
	OFD *fd;
	long rc;

	if (h < 0)
		return(E_OK);	/* always a good close on a character device */

	if ((h0 = h) < NUMSTD)
	{
		h = run->p_uft[h];
		run->p_uft[h0] = 0;	/* mark std dev as not in use */
		if (h < 0)
			return(E_OK);
	}
	else if (((long) sft[h-NUMSTD].f_ofd) < 0L)
	{
		if (!(--sft[h-NUMSTD].f_use))
		{
			sft[h-NUMSTD].f_ofd = 0;
			sft[h-NUMSTD].f_own = 0;
		}
		return(E_OK);
	}

	if (!(fd = getofd(h)))
		return(EIHNDL);

	rc = ixclose(fd,0);

	if (!(--sft[h-NUMSTD].f_use))
	{
		xmfreblk(sft[h-NUMSTD].f_ofd);
		sft[h-NUMSTD].f_ofd = 0;
		sft[h-NUMSTD].f_own = 0;
	}
	return(rc);
}


/*
**  ixclose -
**
**	Error returns
**		EINTRN
**
**	Last modified	SCC	10 Apr 85
**
**	NOTE:	I'm not sure that returning immediatly upon an error from 
**		ixlseek() is the right thing to do.  Some data structures may 
**		not be updated correctly.  Watch out for this!
**		Also, I'm not sure that the EINTRN return is ok.
*/

#define CL_DIR 0x0002	/* this is a directory file, flush, do not free */
#define CL_FULL 0x0004  /* even though its a directory, full close */

long	ixclose(fd,part)
	OFD *fd;
	int part;
{
	DMD *dm;
	OFD *p,**q;
	long tmp;
	int n,i;
	BCB *b;

	dm = fd->o_dmd;

	if (fd->o_flag & O_DIRTY)
	{
		ixlseek(fd->o_dirfil,fd->o_dirbyt+22);

		swp68(fd->o_strtcl);
		swp68l(fd->o_fileln);

		if (part & CL_DIR)
		{
			tmp = fd->o_fileln;
			fd->o_fileln = 0;
			ixwrite(fd->o_dirfil,10L,&fd->o_time);
			fd->o_fileln = tmp;
		}
		else
			ixwrite(fd->o_dirfil,10L,&fd->o_time);

		swp68(fd->o_strtcl);
		swp68l(fd->o_fileln);
	}

	if ((!part) || (part & CL_FULL))
	{
		q = &fd->o_dnode->d_files;

		for (p = *q; p ; p = *(q = &p->o_link))
			if (p == fd)
				break;

		 /* someone else has this file open **** TBA */	/*<<<<<<<<<<<<<<<<<<<<<*/

		if (p)
			*q = p->o_link;
		else
			return(EINTRN);	/* some kind of internal error */	/*<<<<<*/
	}

	/* only flush to appropriate drive ***** TBA ******/	/*<<<<<<<<<<<<<<<<<<<<<*/

	for (i=0; i<2; i++)
		for (b = bufl[i]; b; b = b->b_link)
			flush(b);

	return(E_OK);
}

/*
**  flush -
**
**	Last modified	SCC	10 Apr 85
**
**	NOTE:	rwabs() is a macro that includes a longjmp() which is executed 
**		if the BIOS returns an error, therefore flush() does not need 
**		to return any error codes.
*/

flush(b)
BCB *b;
{
	int n,d;
	DMD *dm;

	/* if buffer not in use or not dirty, no work to do */

	if ((b->b_bufdrv == -1) || (!b->b_dirty))
	{
		b->b_bufdrv = -1;
		return;
	}

	dm = b->b_dm;
	n = b->b_buftyp;
	d = b->b_bufdrv;
	b->b_bufdrv = -1;			/* invalidate in case of error */

	rwabs(1,b->b_bufr,1,b->b_bufrec+dm->m_recoff[n],d);

	/* flush to both fats */

	if (n == 0) 
		rwabs(1,b->b_bufr,1,b->b_bufrec+dm->m_recoff[0]-dm->m_fsiz,d);

	b->b_bufdrv = d;			/* re-validate */
	b->b_dirty = 0;
}

/*
**  usrio -
**
**	Last modified	SCC	10 Apr 85
**
**	NOTE:	rwabs() is a macro that includes a longjmp() which is executed 
**		if the BIOS returns an error, therefore usrio() does not need 
**		to return any error codes.
*/

usrio(rwflg,num,strt,ubuf,dm)
int rwflg,num,strt;
char *ubuf;
DMD *dm;
{
	BCB *b;

	for (b = bufl[1]; b; b = b->b_link)
		if ((b->b_bufdrv == dm->m_drvnum) &&
		   (b->b_bufrec >= strt) &&
		   (b->b_bufrec < strt+num))
			flush(b);

	rwabs(rwflg,ubuf,num,strt+dm->m_recoff[2],dm->m_drvnum);
}


/*
**  getrec -
**	return the ptr to the buffer containing the desired record
*/

char	*getrec(recn,dm,wrtflg)
	int recn;
	int wrtflg;
	DMD *dm;
{
	BCB *b,*p,*mtbuf,**q,**phdr;
	int n,cl,err;

	/* put bcb management here */

	cl = recn >> dm->m_clrlog;	/*  calculate cluster nbr	*/

	if (cl < dm->m_dtl->d_strtcl)
		n = 0;					/* FAT operat'n	*/
	else if (recn < 0)
		n = 1;					/*  DIR (?)	*/
	else
		n = 2;					/*  DATA (?)	*/

	mtbuf = 0;
	phdr = &bufl[(n != 0)];

	/*
	**  see if the desired record for the desired drive is in memory.
	**	if it is, we will use it.  Otherwise we will use
	**		the last invalid (available) buffer,  or
	**		the last (least recently) used buffer.
	*/

	for (b = *(q = phdr); b; b = *(q = &b->b_link))
	{
		if ((b->b_bufdrv == dm->m_drvnum) && (b->b_bufrec == recn))
			break;
		/*  
		**  keep track of the last invalid buffer
		*/
		if (b->b_bufdrv == -1)		/*  if buffer not valid	*/
			mtbuf = b;		/*    then it's 'empty'	*/
	}


	if (!b)
	{	/* 
		**  not in memory.  If there was an 'empty; buffer, use it.
		*/
		if (mtbuf)
			b = mtbuf;

		/*
		**  find predecessor of mtbuf, or last guy in list, which
		**  is the least recently used.
		*/

doio:		for (p = *(q = phdr); p->b_link; p = *(q = &p->b_link))
			if (b == p)
				break;
		b = p;

		/*
		**  flush the current contents of the buffer, and read in the 
		**	new record.
		*/

		flush(b);
		rwabs(0,b->b_bufr,1,recn+dm->m_recoff[n],dm->m_drvnum);

		/*
		**  make the new buffer current
		*/

		b->b_bufrec = recn;
		b->b_dirty = 0;
		b->b_buftyp = n;
		b->b_bufdrv = dm->m_drvnum;
		b->b_dm = dm;
	}
	else
	{	/* use a buffer, but first validate media */
		if (err = trap13(9,b->b_bufdrv))
			if (err == 1)
				goto doio; /* media may be changed */
			else if (err == 2)
			{ /* media definitely changed */
				errdrv = b->b_bufdrv;
				rwerr = E_CHNG; /* media change */
				longjmp(errbuf,rwerr);
			}
	}

	/*
	**  now put the current buffer at the head of the list
	*/

	*q = b->b_link;
	b->b_link = *phdr;
	*phdr = b;

	/*
	**  if we are writing to the buffer, dirty it.
	*/

	if (wrtflg)
		b->b_dirty = 1;

	return(b->b_bufr);
}


/*
**  makeofd -
*/

OFD	*makofd(p)
	DND *p;
{
	OFD *f;

	if (!(f = MGET(OFD))) return (NULPTR);
	f->o_strtcl = p->d_strtcl;
	f->o_fileln = 0x7fffffffL;
	f->o_dirfil = p->d_dirfil;
	f->o_dnode = p->d_parent;
	f->o_dirbyt = p->d_dirpos;
	f->o_date = p->d_date;
	f->o_time = p->d_time;
	f->o_dmd = p->d_drv;
	return(f);
}

/*
**  match() -
**	utility routine to compare file names
**
**	Last modified	SCC	19 Jul 85
*/

int	match(s1,s2)
	char *s1,*s2;
{
	int i;

	if (*s2 == 0xe5)
	{
		if (*s1 == '?')
			return(0);
		else
			if (*s1 == 0xe5)
				return(1);
	}

	for (i=0; i < 11 ; i++, s1++, s2++)
		if (*s1 != '?')
			if (uc(*s1) != uc(*s2))
				return(0);

	/* check attribute match */
	if (*s1 != 8)
		if (!(*s2))
			return(1);

	if (*s1 & *s2)
		return(1);

	return(0);
}

/*	
**  builds -
**
**	Last modified	LTG	23 Jul 85
*/

builds(s1,s2)
	char *s1,*s2; /* s1 is source, s2 dest */
{
	int i;
	char c;

	for (i=0; (i<8) && (*s1) && (*s1 != '*') && (*s1 != SLASH) &&
	    (*s1 != '.') && (*s1 != ' '); i++)
		*s2++ = uc(*s1++);

	if (i == 8)
		while (*s1 && (*s1 != '.') && (*s1 != SLASH))
			s1++;

	c = ((*s1 == '*') ? '?' : ' ');

	if (*s1 == '*')
		s1++;

	if (*s1 == '.')
		s1++;

	for (; i < 8; i++)
		*s2++ = c;

	for (i=0;(i<3) && (*s1) && (*s1 != '*') && (*s1 != SLASH) &&
	    (*s1 != '.') && (*s1 != ' '); i++)
		*s2++ = uc(*s1++);

	c = ((*s1 == '*') ? '?' : ' ');

	for (; i < 3; i++)
		*s2++ = c;
}


/*
**  getpath -
**
**	Last modified	SCC	8 Apr 85
*/

int	getpath(p,d,flg)
	char *p,*d;
	int flg;
{
	int i,i2;
	char *p1;

	for (i=0,p1 = p; *p1; p1++,i++)
		if (*p1 == SLASH)
			break;

	if ((*p1) || flg)
	{
		i2 = 0;
		if (p[0] == '.')
		{
			i2--;
			if (p[1] == '.')
				i2--;
			return(i2);
		}
		else
			if (i)
				builds(p,d);
		return(i);
	}

	return(0);
}


/*
**  xread -
**	read 'len' bytes  from handle 'h'
**
**	Function 0x3F	f_read
**
**	Error returns
**		EIHNDL
**		bios()
**
**	Last modified	SCC	8 Apr 85
*/

long	xread(h,len,ubufr) 
	int h;
	long len;
	char *ubufr;
{
	OFD	*p;

	if (p = getofd(h))
		return(ixread(p,len,ubufr));

	return(EIHNDL);
}

/*
**  ixread -
**
**	Last modified	SCC	26 July 85
*/

long	ixread(p,len,ubufr)
	OFD *p;
	long len;
	char *ubufr;
{
	long maxlen;

        /*Make sure file not opened as write only.*/
        if (p->o_mod == 1)
		return (EACCDN);

	if (len > (maxlen = p->o_fileln - p->o_bytnum))
		len = maxlen;

	if (len > 0)
		return(xrw(0,p,len,ubufr,xfr2usr));

	return(0L);	/* zero bytes read for zero requested */
}


/*
**  xwrite -
**	write 'len' bytes to handle 'h'.
**
**	Function 0x40	f_write
**
**	Error returns
**		EIHNDL
**		bios()
**
**	Last modified	SCC	10 Apr 85
*/

long	xwrite(h,len,ubufr) 
	int h;
	long len;
	char *ubufr;
{
	OFD	*p;

	if (p = getofd(h))
           {
           /* Make sure file not opened as read only.*/
           if (p->o_mod == 0) return (EACCDN);

           return(ixwrite(p,len,ubufr));
           }

	return(EIHNDL);
}

/*
**  ixwrite -
*/

long	ixwrite(p,len,ubufr)
	OFD *p;
	long len;
	char *ubufr;
{
	return(xrw(1,p,len,ubufr,usr2xfr));
}


/*
**  clfix -
**	replace the contents of the fat entry indexed by 'cl' with the value
**	'link', which is the index of the next cluster in the chain.
**
**	Last modified	SCC	14 Apr 85
*/

clfix(cl,link,dm)
	int cl,link;
	DMD *dm;
{
	int f[1],mask;
	long pos;

	if (dm->m_16)
	{
		swp68(link);
		pos = cl << 1;
		ixlseek(dm->m_fatofd,pos);
		ixwrite(dm->m_fatofd,2L,&link);
		return;
	}

	pos = (cl + (cl >> 1));

	link = link & 0x0fff;

	if (cl & 1)
	{
		link = link << 4;
		mask = 0x000f;
	}
	else
		mask = 0xf000;

	ixlseek(dm->m_fatofd,pos);

	/* pre -read */
	ixread(dm->m_fatofd,2L,f);

	swp68(f[0]);
	f[0] = (f[0] & mask) | link;
	swp68(f[0]);

	ixlseek(dm->m_fatofd,pos);
	ixwrite(dm->m_fatofd,2L,f);
}


/*
**  getcl -
**	get the contents of the fat entry indexed by 'cl'.
**
**  returns
**	-1 if entry contains the end of file marker
**	otherwise, the contents of the entry (16 bit value always returned).
**
**	Last modified	SCC	14 Apr 85
*/

int	getcl(cl,dm)
	int cl;
	DMD *dm;
{
	unsigned f[1];

	if (cl < 0)
		return(cl+1);

	if (dm->m_16)
	{
		ixlseek(dm->m_fatofd,((long) (cl << 1)));
		ixread(dm->m_fatofd,2L,f);
		swp68(f[0]);
		return(f[0]);
	}

	ixlseek(dm->m_fatofd,((long) (cl + (cl >> 1))));
	ixread(dm->m_fatofd,2L,f);
	swp68(f[0]);

	if (cl & 1)
		cl = f[0] >> 4;
	else
		cl = 0x0fff & f[0];

	if (cl == 0x0fff)
		return(-1);

	return(cl);
}

/*
**  nextcl -
**	get the cluster number which follows the cluster indicated in the curcl
**	field of the OFD, and place it in the OFD.
**
**  returns
**	E_OK	if success,
**	-1	if error
*/

nextcl(p,wrtflg)
	OFD *p;
	int wrtflg;
{
	DMD *dm;
	int i,cl,cl2,rover;

	cl = p->o_curcl;
	dm = p->o_dmd;

	if (cl < 0)
	{
		cl2 = cl + 1;
		goto retcl;
	}

	if (cl > 0)
		cl2 = getcl(cl,dm);

	if (cl == 0)
		cl2 = (p->o_strtcl ? p->o_strtcl : -1);

	if (wrtflg && (cl2 == -1))
	{ /* end of file, allocate new clusters */
		rover = cl;
		for (i=2; i < dm->m_numcl; i++) 
		{
			if (rover < 2)
				rover = 2;

			if (!(cl2 = getcl(rover,dm)))
				break;
			else
				rover = (rover + 1) % dm->m_numcl;
		}

		cl2 = rover;

		if (i < dm->m_numcl)
		{
			clfix(cl2,-1,dm);
			if (cl)
				clfix(cl,cl2,dm);
			else
			{
				p->o_strtcl = cl2;
				p->o_flag |= O_DIRTY;
			}
		}
		else
			return(-1);
	}

	if (cl2 == -1)
		return(-1);

retcl:	p->o_curcl = cl2;
	p->o_currec = cl2rec(cl2,dm);
	p->o_curbyt = 0;

	return(E_OK);
}

/*
**  addit -
**	update the OFD for the file to reflect the fact that 'siz' bytes
**	have been written to it.
*/

addit(p,siz,flg)
	OFD *p;
	long siz;
	int flg; /* update curbyt ? (yes if less than 1 cluster transferred) */
{
	p->o_bytnum += siz;

	if (flg)
		p->o_curbyt += siz;

	if (p->o_bytnum > p->o_fileln)
	{
		p->o_fileln = p->o_bytnum;
		p->o_flag |= O_DIRTY;
	}
}


/*
**  xrw -
**	read/write 'len' bytes from/to the file indicated by the OFD at 'p'.
**
**  details
**	We wish to do the i/o in whole clusters as much as possible.
**	Therefore, we break the i/o up into 5 sections.  Data which occupies 
**	part of a logical record (e.g., sector) with data not in the request 
**	(both at the start and the end of the the request) are handled
**	separateley and are called header (tail) bytes.  Data which are
**	contained complete in sectors but share part of a cluster with data not
**	in the request are called header (tail) records.  These are also
**	handled separately.  In between handling of header and tail sections,
**	we do i/o in terms of whole clusters.
**
**  returns
**	nbr of bytes read/written from/to the file.
*/

long	xrw(wrtflg,p,len,ubufr,bufxfr)
	int wrtflg,(*bufxfr)();
	OFD *p;
	long len;
	char *ubufr;
{
	char *bufp;
	DMD *dm;
	int bytn,recn,rsiz,lenxfr,csiz,csizb,curcl,lentail,num;
	int hdrrec,lsiz,tailrec;
	int last, nrecs, lflg; /* multi-sector variables */
	long nbyts;
	long rc,bytpos,lenrec,lenmid;

	/*
	**  determine where we currently are in the filef
	*/

	dm = p->o_dmd;			/*  get drive media descriptor	*/
	rsiz = dm->m_recsiz;		/*	record size		*/
	csiz = dm->m_clsiz;		/*	cluster size		*/
	csizb = dm->m_clsizb;		/*	cluster size in bytes	*/

	bytpos = p->o_bytnum; 		/*  starting file position 	*/
	curcl = p->o_curcl;		/*  current cluster number	*/

	/*
	**  get logical record number to start i/o with
	**	(bytn will be byte offset into sector # recn)
	*/

	recn = divmod(&bytn,(long) p->o_curbyt,dm->m_rblog);
	recn += p->o_currec;

	/*
	**  determine "header" of request.
	*/

	if (bytn) /* do header */
	{	/*
		**  xfer len is
		**	min( #bytes req'd , #bytes left in current record )
		*/
		lenxfr = min(len,rsiz-bytn);
		bufp = getrec(recn,dm,wrtflg);	/*  get desired record	*/
		addit(p,(long) lenxfr,1);	/*  update ofd		*/
		len -= lenxfr;			/*  nbr left to do	*/
		recn++;				/*    starting w/ next	*/

		if (!ubufr) 
		{
			rc = (long) (bufp+bytn);	/* ???????????	*/
			goto exit;
		}

		(*bufxfr)(lenxfr,bufp+bytn,ubufr);	
		ubufr += lenxfr;
	}

	/*
	**  "header" complete.  See if there is a "tail".  
	**  After that, see if there is anything left in the middle.
	*/

	lentail = len & dm->m_rbm;

	if (lenmid = len - lentail)		/*  Is there a Middle ?	*/
	{ 	
		hdrrec = recn & dm->m_clrm;

		if (hdrrec)
		{
			/*  if hdrrec != 0, then we do not start on a clus bndy;
			**	so determine the min of (the nbr sects 
			**	remaining in the current cluster) and (the nbr 
			**	of sects remaining in the file).  This will be 
			**	the number of header records to read/write.
			*/
			hdrrec = ( dm->m_clsiz - hdrrec ) ;	/* M00.14.01 */
			if( hdrrec > lenmid >> dm->m_rblog )	/* M00.14.01 */
				hdrrec = lenmid >> dm->m_rblog;	/* M00.14.01 */
			usrio(wrtflg,hdrrec,recn,ubufr,dm);
			ubufr += (lsiz = hdrrec << dm->m_rblog);
			lenmid -= lsiz;
			addit(p,(long) lsiz,1);
		}

		/* 
		**  do whole clusters 
		*/

		lenrec = lenmid >> dm->m_rblog;		   /* nbr of records  */
		num = divmod(&tailrec,lenrec,dm->m_clrlog);/* nbr of clusters */

		last = nrecs = nbyts = lflg = 0;

		while (num--)		/*  for each whole cluster...	*/
		{
			rc = nextcl(p,wrtflg);

			/* 
			**  if eof or non-contiguous cluster, or last cluster 
			**	of request, 
			**	then finish pending I/O 
			*/

			if ((!rc) && (p->o_currec == last + nrecs))
			{
				nrecs += csiz;
				nbyts += csizb;
				if (!num) goto mulio;
			}
			else
			{
				if (!num)
					lflg = 1;
mulio:				if (nrecs)
					usrio(wrtflg,nrecs,last,ubufr,dm);
				ubufr += nbyts;
				addit(p,nbyts,0);
				if (rc)
					goto eof;
				last = p->o_currec;
				nrecs = csiz;
				nbyts = csizb;
				if ((!num) && lflg)
				{
					lflg = 0;
					goto mulio;
				}
			}
		}  /*  end while  */

		/* 
		**  do "tail" records 
		*/

		if (tailrec)
		{
			if (nextcl(p,wrtflg))
				goto eof;
			lsiz = tailrec << dm->m_rblog;
			addit(p,(long) lsiz,1);
			usrio(wrtflg,tailrec,p->o_currec,ubufr,dm);
			ubufr += lsiz;
		}
	}

	/* 
	**	do tail bytes within this cluster 
	*/

	if (lentail)
	{
		recn = divmod(&bytn,(long) p->o_curbyt,dm->m_rblog);

		if((!recn) || (recn == csiz))
		{
			if (nextcl(p,wrtflg))
				goto eof;
			recn = 0;
		}

		bufp = getrec(p->o_currec+recn,dm,wrtflg);
		addit(p,(long) lentail,1);

		if (!ubufr)
		{
			rc = (long) bufp;
			goto exit;
		}

		(*bufxfr)(lentail,bufp,ubufr,wrtflg);
	} /*  end tail bytes  */

eof:	rc = p->o_bytnum - bytpos;
exit:	return(rc);

}

/*
**  makdnd -
**	make a child subdirectory of directory p
*/

DND	*makdnd(p,b) 
	DND *p;
	FCB *b;
{
	OFD *fd;
	DND *p1;

	fd = p->d_ofd;
	if (!(p1 = MGET(DND))) return (NULPTR);

	if (p->d_left)
		p1->d_right = p->d_left;

	p->d_left = p1;
	p1->d_parent = p;
	p1->d_ofd = NULPTR;
	p1->d_strtcl = b->f_clust;
	swp68(p1->d_strtcl);
	p1->d_drv = p->d_drv;
	p1->d_dirfil = fd;
	p1->d_dirpos = fd->o_bytnum - 32;
	p1->d_time = b->f_time;
	p1->d_date = b->f_date;
	xmovs(11,b->f_name,p1->d_name);

	return(p1);
}


/*	
**  scan -
**
**	Last modified	SCC	22 Jul 85
*/

FCB	*scan(p,n,att,posp)
	DND *p;
	long *posp;
	int att;
	char *n;
{
	OFD *fd;
	FCB *b;
	DND *p1;
	int m, got_it;
	long srchpos;
	char d[12];

	m = 0;
	builds(n,d);
	d[11] = att;

	if (!(fd = p->d_ofd))
		if (!(p->d_ofd = (fd = makofd(p))))
			return (NULPTR);

        /* not directory search */
	if ((srchpos = *posp) == -1)
		srchpos = p->d_scan;

	ixlseek(fd,srchpos);

	for (got_it = 0, p1 = p->d_left; p1; p1 = p1->d_right)
		if (got_it = xcmps(d,p1->f_name))
			break;

	while ((b = (FCB *) ixread(fd,32L,NULPTR)) && (b->f_name[0]))
	{
		/* completed dir traversal or create subdir DND ? */

		if ((fd->o_bytnum > p->d_scan) && (!(fd->o_flag & O_COMPLETE)))
			/* don't add DND if . or .. or erased or have this one */
			if ((b->f_name[0] != '.') && (b->f_attrib & FA_SUBDIR) &&
			    (b->f_name[0] != 0xE5) && !(xcmps(d,b->f_name) && got_it) )
				if (!(p1 = makdnd(p,b)))
					return (NULPTR);

		if (m = match(d,b->f_name))
			break;
	}

	/* restore directory scanning pointer */
	if (*posp == -1)
	{
		if (fd->o_bytnum > p->d_scan)
			p->d_scan = fd->o_bytnum;
	}
	else
		*posp = fd->o_bytnum;

	if (!m)
	{
		if (b && (*n == 0xe5))
			return(b);
		fd->o_flag |= O_COMPLETE;
		return(NULPTR);
	}

	if (*posp == -1)
	{
		ixlseek(fd,fd->o_bytnum - 32);

		return(((FCB *) p1));
	}

	return(b);
}


/*
**  ckdrv -
**
**	Last modified	SCC	8 Apr 85
*/

long	ckdrv(d)
	int d; /* has this drive been accessed, or had a media change */
{
	int mask,i;
	BPB *b;

	mask = 1 << d;

	if (!(mask & drvsel))
	{
		b = getbpb(d);
		if (!b)
			return(ERROR);
		if (log(b,d)) return (ENSMEM);
		drvsel |= mask;
	}

	if ((!run->p_curdir[d]) || (!dirtbl[run->p_curdir[d]]))
	{  /* need to allocate current dir on this drv */
		for (i = 1; i < NCURDIR; i++)
			if (!diruse[i])
				break;

		if (i == NCURDIR)
			return(ERROR); /* barf and gag *** TBA ***/	/*<<<<<<<<<<<<<*/

		diruse[i]++;
		dirtbl[i] = drvtbl[d]->m_dtl;
		run->p_curdir[d] = i;
	}
	return(d);
}

/*	
**  dcrack -
**
**
**	Last modified	19 Jul 85	SCC
*/

DND	*dcrack(np)
	char **np;
{
	char *n;
	int d;
	DND *p;

	/* check current directory records */
	n = *np;
	if (n[1] == ':')
	{
		d = uc(n[0]) - 'A';
		n += 2;
	}
	else
		d = run->p_curdrv;

	if (ckdrv(d) < 0)
		return(0L);

	if (*n == SLASH)
	{ /* [D:]\path */
		p = drvtbl[d]->m_dtl;
		n++;
	}
	else
		p = dirtbl[run->p_curdir[d]];

	/* whew ! */
	*np = n;
	return(p);
}

/*	
**  findit -
**
**	Last modified	SCC	19 Jul 85
*/

DND	*findit(name,sp,dflag)
	char *name;
	char **sp;
	int dflag; /* if true, no file name is present */
{
	char *n;
	DND *p,*pp,*newp;
	DMD *dm;
	int i;
	char s[11];

	/* crack directory and drive */

	n = name;

	if (!(p = dcrack(&n)))
		return(p);

	do
	{
		if (!(i = getpath(n,s,dflag)))
			break;

		if (i < 0)
		{
			if (i == -2)
				p = p->d_parent;
			i = -i;
			goto scanxt;
		}

		if (!(newp = p->d_left))
			newp = dirscan(p,n);

		pp = p;

		if (!(p = newp))
			break;

		/* check all subdirectories at this level */

		while (p && (!xcmps(s,p->d_name))) 
		{  /* if dir not all traversed, scan until match or end  */
			if (!(newp = p->d_right))
			{
			  p = 0;
			  if (pp)
			    if (!(pp->d_ofd->o_flag & O_COMPLETE))
				p = dirscan(pp,n);
			}
			else
				p = newp;
		}
scanxt:		if (*(n = n + i))
			n++;
		else
			break;
	} while (p && i);

	/* p = 0 ==> not found
	   i = 0 ==> found at p (dnd entry)
	   n = points at filename */

	*sp = n;
	return(p);
}


/*	
**  xchdir -
**	change current dir to path p (extended cd n:=[a:][\bin])
**
**	Function 0x3B	d_setpath
**
**	Error returns
**		EPTHNF
**		ckdrv()
**
**	Last modified	LTG	6 Aug 85
**
**	SCC	The logical drive stuff is not fully implemented and does not
**		currently work.
*/

long	xchdir(p) 
	char *p;
{
	long l;
	int dphy,dr,dlog,i,flg;
	char *s;

	flg = 1;

xch:	if (p[1] == ':')
		dphy = uc(p[0]) - 'A';
	else
		dphy = run->p_curdrv;

	if (flg)
	{
		dlog = dphy;
		if (p[2] == '=')
		{
			flg = 0;
			p += 3;
			goto xch;
		}
	}

	if ((l=ckdrv(dphy)) < 0)
		return(l);

	/* find space in dirtbl */
	if (dr = run->p_curdir[dlog])
	{
		--diruse[dr]; /* someone is still using it */
	}

	for (i = 0; i < NCURDIR; i++, dr++)
	{
		if (dr == NCURDIR)
			dr = 0;
		if (!diruse[dr])
			break;
	}

	if (i == NCURDIR)
		return(EPTHNF);

	diruse[dr]++;

	if (!(l = findit(p,&s,1)))
		return(EPTHNF);

	dirtbl[dr] = l;

	run->p_curdir[dlog] = dr;

	return(E_OK);
}


/*
**  packit -
**	pack into user buffer
*/

char	*packit(s,d)
	char *s,*d;
{ 
	char *s0;
	int i;

	if (!(*s))
		goto pakok;

	s0 = s;
	for (i=0; (i < 8) && (*s) && (*s != ' '); i++)
		*d++ = *s++;

	if (*s0 == '.')
		goto pakok;

	s = s0 + 8; /* ext */

	if (*s != ' ')
		*d++ = '.';
	else
		goto pakok;

	for (i=0; (i < 3) && (*s) && (*s != ' '); i++)
		*d++ = *s++;
pakok:	*d = 0;
	return(d);
}


/*
**  dopath -
*/

char	*dopath(p,buf)
	DND *p;
	char *buf;
{
	if (p->d_parent)
		buf = dopath(p->d_parent,buf);
	buf = packit(p->d_name,buf);
	*buf++ = SLASH;
	return(buf);
}


/*	
**  xgetdir -
**
**	Function 0x47	d_getpath
**
**	Error returns
**		EDRIVE
**
**	Last modified	SCC	11 Apr 85
*/

long	xgetdir(buf,drv) /*+ return text of current dir into specified buffer */
	int drv;
	char *buf;
{
	DND *p;

	drv = (drv ? drv-1 : run->p_curdrv);

	if (ckdrv(drv) < 0)
	{
		*buf = 0;
		return(EDRIVE);
	}

	p = dirtbl[run->p_curdir[drv]];
	buf = dopath(p,buf);
	*--buf = 0;	/* null as last char, not slash */

	return(E_OK);
}


/*
**  xgetdta -
**	Function 0x2F	f_getdta
*/

char
*xgetdta() /*+ return address of dta */
{
	return(run->p_xdta);
}


/*
**  xsetdta -
**
**	Function 0x1A	f_setdta
*/

xsetdta(addr) /*+ set transfer address to addr */
	char *addr;
{
	run->p_xdta = addr;
}


/*
**  xsetdrv 0
**	set default drive
**	( 0 = A, etc )
**	Function 0x0E	d_setdrv
*/

long	xsetdrv(drv) 
{
	run->p_curdrv = drv;
	return( trap13(0x0A) );					
}


/*
**  xgetdrv -
**	get default drive
**	(0 = A, etc )
**
**	Function 0x19	d_getdrv
**
**	Last modified	SCC	1 May 85
*/

long	xgetdrv() 
{
	return(run->p_curdrv);
}


/*
**  xsfirst -
**	search first for matching name, into dta
**
**	Function 0x4E	f_sfirst
**
**	Error returns
**		EFILNF
**
**	Last modified	SCC	6 May 85
*/

long	xsfirst(name,att) 
	char *name;
	int att;
{
	return(ixsfirst(name,att,run->p_xdta));
}

/* search first for matching name, into specified address */
/* if address = 0L, caller wants search only, no buffer info */

/*
**  ixsfirst -
**
*/

long	ixsfirst(name,att,addr)
	char *name;
	int att;
	char *addr;
{
	char *s,a[11];
	DND *dn;
	FCB *f;
	long pos;

	if (att != 8)
		att |= 0x21;

	if (!(dn = findit(name,&s,0)))
		return(EFILNF);

 /* now scan for filename from start of directory */

	pos = 0;

	if (dn)
	{
		if (!(f = scan(dn,s,att,&pos)))
			return(EFILNF);
	}
	else
		return(EFILNF);

	if (addr)
	{
		xmovs(12,s,addr);
		*(addr + 12) = att;
		xmovs(4,&pos,addr+13);
		xmovs(sizeof(DND *),&dn,addr+17);
		makbuf(f,addr);
	}

	return(E_OK);
}


/*
**  xsnext -
**	search next, return into dta 
**
**	Function 0x4F	f_snext
**
**	Error returns
**		ENMFIL
**
**	Last modified	SCC	10 Apr 85
*/

long	xsnext() 
{
	DND *dn;
	char att;
	long pos;
	FCB *f;

	xmovs(sizeof(DND *),run->p_xdta+17,&dn);
	xmovs(4,run->p_xdta+13,&pos);
	att = *(run->p_xdta + 12);

	if (!(f = scan(dn,run->p_xdta,att,&pos)))
		return(ENMFIL);

	xmovs(4,&pos,run->p_xdta+13);
	makbuf(f,run->p_xdta);

	return(E_OK);
}


/*
**  makbuf -
*/

makbuf(f,ubufr)
	FCB *f;
	char *ubufr;
{
	char *s,*d;
	int i;

	*(ubufr+21) = f->f_attrib;
	xmovs(4,&f->f_time,ubufr+22);
	swp68(ubufr[22]);
	swp68(ubufr[24]);
	xmovs(4,&f->f_fileln,ubufr+26);
	swp68l(ubufr[26]);
	packit(f,ubufr+30);
}


/*
**  opnfil -
**
**	Error returns
**		ENHNDL
**
**	Last modified	SCC	8 Apr 85
*/

long	opnfil(f, dn, mod)
	FCB *f;
	DND *dn;
	int mod;
{
	int i,h;

	/* find free sft handle */
	for (i = 0; i < OPNFILES; i++)
		if (!sft[i].f_own)
			break;

	if (i == OPNFILES)
		return(ENHNDL);

	sft[i].f_own = run;
	sft[i].f_use = 1;
	h = i+NUMSTD;

	return(makopn(f, dn, h, mod));
}


/*
**  makopn -
**	make an open file for sft handle h 
**
**	Last modified	SCC	8 Apr 85
*/

long	makopn(f, dn, h, mod) 
	FCB *f;
	DND *dn;
	int h;
	int mod;
{
	OFD *p,*p2;
	DMD *dm;
	int sh;

	dm = dn->d_drv;

	if (!(p = MGET(OFD))) return (ENSMEM);

        /* set mode */
        p->o_mod = mod;
	p->o_dmd = dm;
	sft[h-NUMSTD].f_ofd = p;
	p->o_usecnt = 0;
	p->o_curcl = 0;
	p->o_curbyt = 0;
	p->o_dnode = dn;
	p->o_dirfil = dn->d_ofd;
	p->o_dirbyt = dn->d_ofd->o_bytnum - 32;

	for (p2 = dn->d_files; p2; p2 = p2->o_link)
		if (p2->o_dirbyt == p->o_dirbyt)
			break; /* same dir, same dcnt */

	p->o_link = dn->d_files;
	dn->d_files = p;

	if (p2)
	{ /* steal time,date,startcl,fileln */
		xmovs(12,&p2->o_time,&p->o_time);
		p2->o_thread = p; /* not used yet... TBA *********/	/*<<<<<<<<<<<<<*/
	}
	else
	{
		p->o_strtcl = f->f_clust;
		swp68(p->o_strtcl);
		p->o_fileln = f->f_fileln;
		swp68l(p->o_fileln);
		p->o_date = f->f_date;
		p->o_time = f->f_time;
	}

	return(h);
}


/*
**  dirinit -
*/

FCB	*dirinit(dn)
	DND *dn;
{
	OFD *fd;
	int num,i,i2;
	char *s1;
	DMD *dm;
	FCB *f1;

	fd = dn->d_ofd;
	num = (dm = fd->o_dmd)->m_recsiz;
	for (i2 = 1; i2 < dm->m_clsiz; i2++)
	{
		s1 = getrec(fd->o_currec+i2,dn->d_drv,1);
		for (i = 0; i < num; i++)
			*s1++ = 0;
	}
	f1 = (FCB *) (s1 = getrec(fd->o_currec,dn->d_drv,1));
	for (i = 0; i < num; i++)
		*s1++ = 0;
	return(f1);
}


/*
**  xcreat -
**  create file with specified name, attributes
**
**	Function 0x3C	f_create
**
**	Error returns
**		EPTHNF
**		EACCDN
**		ENHNDL
**
**	Last modified	SCC	13 May 85
*/

long	xcreat(name,attr) 
	char *name;
	char attr;
{
	return(ixcreat(name, attr & 0xEF));
}

/*	Last modified	SCC	29 Jul 85		### ktb 
*/

long
ixcreat(name,attr)
char *name;
char attr;
{
	DND *dn;
	FCB *f;
	OFD *fd;
	char *s,*s1,n[2],a[11];
	int i,num,f2;
	long pos,rc;

	n[0] = 0xe5; n[1] = 0;

 /* first find path */

	if (!(dn = findit(name,&s,0)))
		return(EPTHNF);

	if (!*s || (*s == '.'))
		return(EPTHNF);

	if (!(fd = dn->d_ofd))
		if (!(dn->d_ofd = (fd = makofd(dn)))) return (ENSMEM);

 /* is it already there ? */

	pos = 0;

	if (f = scan(dn,s,-1,&pos))
	{
		if (f->f_attrib & (FA_SUBDIR | FA_RO))
			return(EACCDN);
		pos -= 32;
		ixdel(dn,f,pos);
	}
	else
		pos = 0;

 /* now scan for empty space */

	if (!(f = scan(dn,n,-1,&pos)))
	{
		/* need to grow the directory */
		if (fd->o_curcl < 0)
			return(EACCDN);				/* root is full */

		if (nextcl(fd,1))
			return(EACCDN);

		f = dirinit(dn);
	}

	builds(s,a);
	pos -= 32;
	f->f_attrib = attr;
	for (i=0; i<10; i++)
		f->f_fill[i] = 0;
	f->f_time = time;
	swp68(f->f_time);
	f->f_date = date;
	swp68(f->f_date);
	f->f_clust = 0;
	f->f_fileln = 0;
	ixlseek(fd,pos);
	ixwrite(fd,11L,a);	/* write name, set dirty flag */
	ixclose(fd,CL_DIR);	/* partial close to flush */
	ixlseek(fd,pos);
	s = ixread(fd,32L,NULPTR);
	f2 = rc = opnfil(s,dn, ((f->f_attrib & FA_RO) ? 0 : 2));

	if (rc < 0)
		return(rc);

	getofd(f2)->o_flag |= O_DIRTY;

	return(f2);
}


/*	Function 0x39	d_create

	Last modified	SCC	13 May 85
*/

long
xmkdir(s) /*+ make a directory, path s */
char *s;
{
	int h,cl;
	long rc;
	OFD *f,*fd,*f0;
	FCB *b,*f2;
	DND *dn;

	if ((h = rc = ixcreat(s,FA_SUBDIR)) < 0)
		return(rc);

	f = getofd(h);

	/* build a DND in the tree */

	fd = f->o_dirfil;
	ixlseek(fd,f->o_dirbyt);
	b = (FCB *) ixread(fd,32L,NULPTR);
	if (!(dn = makdnd(f->o_dnode,b))) return (ENSMEM);
	if (!(dn->d_ofd = f0 = makofd(dn))) return (ENSMEM);

	/* initialize dir cluster */

	if (nextcl(f0,1))
		return(EACCDN);

	f2 = dirinit(dn);				/* pointer to dirty dir block */

	/* write identifier */

	xmovs(22,dots,f2);
	f2->f_attrib = FA_SUBDIR;
	f2->f_time = time;
	f2->f_date = date;
	cl = f0->o_strtcl;
	swp68(cl);
	f2->f_clust = cl;
	f2->f_fileln = 0;
	f2++;

	/* write parent entry .. */

	xmovs(22,dots2,f2);
	f2->f_attrib = FA_SUBDIR;
	f2->f_time = time;
	f2->f_date = date;
	cl = f->o_dirfil->o_strtcl;

	if (cl < 0)
		cl = 0;

	swp68(cl);
	f2->f_clust = cl;
	f2->f_fileln = 0;
	xmovs(sizeof(OFD),f0,f);
	f->o_flag |= O_DIRTY;
	ixclose(f,CL_DIR | CL_FULL);	/* force flush and write */
	xmfreblk(f);
	sft[h-NUMSTD].f_own = 0;
	sft[h-NUMSTD].f_ofd = 0;
	return(E_OK);
}


/*	Function 0x3D	f_open

	Error returns
		EFILNF
		opnfil()

	Last modified	SCC	5 Apr 85
*/

long
xopen(name,mod) /*+ open a file, path name -- return a handle */
char *name;
int mod;		
{
	return (ixopen (name, mod));
}

long
ixopen(name, mod)
char *name;
int mod;
{
	FCB *f;
	DND *dn;
	char *s;
	long pos;

        /* first find path */
	if (!(dn = findit(name,&s,0)))
		return(EFILNF);

        /* now scan for filename */
	pos = 0;
	if (!(f = scan(dn,s,FA_NORM,&pos)))
		return(EFILNF);

        /* Check to see if the file is read only*/
        if ((f -> f_attrib & 1) && (mod != 0))
        	return (EACCDN);

       	return (opnfil (f, dn, mod));
}


/*	Function 0x43	f_attrib

	Error returns
		EPTHNF
		EFILNF

	Last modified	SCC	11 Apr 85
*/

char
xchmod(p,wrt,mod) /*+ change/get attrib of path p, wrt 1 is set */
char *p,mod;
int wrt;
{
	OFD *fd;
	DND *dn;
	FCB *f;
	char *s;
	long pos;

	if (!(dn = findit(p,&s,0)))
		return(EPTHNF);

	pos = 0;

	if (!(f = scan(dn,s,FA_NORM,&pos)))
		return(EFILNF);

	pos -= 21;					/* point at attribute in file */
	fd = dn->d_ofd;
	ixlseek(fd,pos);
	if (!wrt)
		ixread(fd,1L,&mod);
	if (wrt)
	{
		ixwrite(fd,1L,&mod);
		ixclose(fd,CL_DIR); /* for flush */
	}
	return(mod);
}


/*	Function 0x57	f_datime
*/

long
xgsdtof(buf,h,wrt) /*+ get/set date/time of file into or from buffer buf */
int h,wrt;
int *buf;
{
	OFD *f,*fd;

	f = getofd(h);
	fd = f->o_dirfil;
	ixlseek(fd,f->o_dirbyt + 22);
	if (!wrt)
		ixread(fd,4L,buf);
	swp68(buf[0]);
	swp68(buf[1]);
	if (wrt)
	{
		ixwrite(fd,4L,buf);
		ixclose(fd,CL_DIR);
	}
}


/*	Function 0x41 	f_delete

	Error returns
		EFILNF

	Last modified	SCC	5 Apr 85
*/

long
xunlink(name) /*+ delete file -- path specified by name */
char *name;
{
	DND *dn;
	char *s;
	long pos;
	FCB *f;

 /* first find path */

	if (!(dn = findit(name,&s,0)))
		return(EFILNF);

 /* now scan for filename */

	pos = 0;
	if (!(f = scan(dn,s,FA_NORM,&pos)))
		return(EFILNF);

	if (f->f_attrib & FA_RO)
		return(EACCDN);

	pos -= 32;

	return(ixdel(dn,f,pos));
}


/*
	Used by
		ixcreat()
		xunlink()
		xrmdir()

	Last modified	SCC	16 Apr 85
*/

long
ixdel(dn,f,pos)
DND *dn;
FCB *f;
long pos;
{
	DMD *dm;
	int n, n2;
	OFD *fd;
	char c;

/*
Traverse the list of files open for this directory node.

If a file is found that has the same position in the directory as the one we are to
delete, then scan the system file table to see if this process is then owner.  If so,
then close it, otherwise abort.

NOTE	that both 'for' loops scan for the entire length of their respective data
	structures, and do not drop out of the loop on the first occurence of a match.
*/

	for (fd = dn->d_files; fd; fd = fd->o_link)
		if (fd->o_dirbyt == pos)
			for (n = 0; n < OPNFILES; n++)
				if (sft[n].f_ofd == fd)
				{
					if (sft[n].f_own == run)
						ixclose(fd,0);
					else
						return(EACCDN);
				}
/*
? Traverse this file's chain of allocated clusters, freeing them.
*/

	dm = dn->d_drv;
	n = f->f_clust;
	swp68(n);

	while (n && (n != -1))
	{
		n2 = getcl(n,dm);
		clfix(n,0,dm);
		n = n2;
	}

/*
? Mark the directory entry as erased.
*/

	fd = dn->d_ofd;
	ixlseek(fd,pos);
	c = 0xe5;
	ixwrite(fd,1L,&c);
	ixclose(fd,CL_DIR);

/*
NOTE	that the preceding routines that do physical disk operations will 'longjmp' on
	failure at the BIOS level, thereby allowing us to simply return with E_OK.
*/

	return(E_OK);
}


/*	Function 0x3A	d_delete

	Error returns
		EPTHNF
		EACCDN
		EINTRN

	Last modified	SCC	29 Jul 85
*/

long
xrmdir(p)
char *p;
{
	DND *d,*d1,**q;
	FCB *f;
	OFD *fd,*f2,*f1,*fx;
	long pos;
	char *s;

	if (!(d = findit(p,&s,1)))
		return(EPTHNF);

	if (*s == '.')			/* Can't delete '.', because we're in it */
		return(EACCDN);		/* Can't delete '..', because it's our parent */

	if (!(fd = d->d_ofd))
		if (!(fd = makofd(d))) return (ENSMEM);

	ixlseek(fd,0x40L);
	do
	{
		if (!(f = (FCB *) ixread(fd,32L,NULPTR)))
			break;
	} while (f->f_name[0] == 0x0e5);

	if (f)
		if (f->f_name[0] != 0)
			return(EACCDN);

	for (d1 = *(q = &d->d_parent->d_left); d1 != d; d1 = *(q = &d1->d_right))
		; /* follow sib-links */

	if (d1 != d)
		return(EINTRN);				/* internal error */

	if (d->d_files)
		return(EINTRN);			/* open files ? - internal error */

	if (d->d_left)
		return(EINTRN);				/* subdir - internal error */

	/* take him out ! */

	*q = d->d_right;

	if (d->d_ofd)
		xmfreblk(d->d_ofd);		

	d1 = d->d_parent;
	xmfreblk(d);
	ixlseek((f2 = fd->o_dirfil),(pos = fd->o_dirbyt));
	f = (FCB *) ixread(f2,32L,NULPTR);

	return(ixdel(d1,f,pos));
}


/*	Function 0x36	d_free

	Error returns
		ERROR

	Last modified	SCC	15 May 85
*/

long
xgetfree(buf,drv) /*+ get disk free space data into buffer */
int drv;
long *buf;
{
	int i,free;
	DMD *dm;

	drv = (drv ? drv-1 : run->p_curdrv);

	if ((i = ckdrv(drv)) < 0)
		return(ERROR);

	dm = drvtbl[i];
	free = 0;
	for (i = 2; i < dm->m_numcl; i++) if (!getcl(i,dm)) free++;
	*buf++ = (long)(free);
	*buf++ = (long)(dm->m_numcl);
	*buf++ = (long)(dm->m_recsiz);
	*buf++ = (long)(dm->m_clsiz);
	return(E_OK);
}


/*	Function 0x56	f_rename

	Error returns
		EPTHNF

	Last modified 	LTG 	26 Jul 85
*/

long
xrename(n,p1,p2)			/*+ rename file, old path p1, new path p2 */
int n;
char *p1,*p2;
{
	char *s1,*s2;
	char buf[11];
	int hnew,att;
	FCB *f;
	DND *dn1,*dn2;
	OFD *f1,*fd,*fd2;
	long rc, h1;

	if (!ixsfirst(p2,0,0L))
		return(EACCDN);

	if (!(dn1 = findit(p1,&s1,0)))
		return(EPTHNF);

	if (!(dn2 = findit(p2,&s2,0)))
		return(EPTHNF);

	if ((h1 = xopen(p1, 2)) < 0L)
		return (h1);
	f1 = getofd ((int)h1);

	fd = f1->o_dirfil;
	buf[0] = 0xe5;
	ixlseek(fd,f1->o_dirbyt);

	if (dn1 != dn2)
	{
		/* get old attribute */
		f = (FCB *) ixread(fd,32L,NULPTR);
		att = f->f_attrib;
		/* erase (0xe5) old file */
		ixlseek(fd,f1->o_dirbyt);
		ixwrite(fd,1L,buf);

		/* copy time/date/clust, etc. */

		ixlseek(fd,f1->o_dirbyt + 22);
		ixread(fd,10L,buf);
		hnew = xcreat(p2,att);
		fd2 = getofd(hnew);
		ixlseek(fd2->o_dirfil,fd2->o_dirbyt + 22);
		ixwrite(fd2->o_dirfil,10L,buf);
		fd2->o_flag &= ~O_DIRTY;
		xclose(hnew);
		ixclose(fd2->o_dirfil,CL_DIR);
	}
	else
	{
		builds(s2,buf);
		ixwrite(fd,11L,buf);
	}

	if ((rc = xclose((int)h1)) < 0L)
		return(rc);

	return(ixclose(fd,CL_DIR));
}


/*	Function 0x42	f_seek

	Error returns
		EIHNDL
		EINVFN
		ixlseek()
*/

long
xlseek(n,h,flg) /*+ seek file handle h, byte position n */
int h,flg;
long n;
{
	OFD *f;

	if (!(f = getofd(h)))
		return(EIHNDL);

	if (flg == 2)
		n += f->o_fileln;
	else if (flg == 1)
		n += f->o_bytnum;
	else if (flg)
		return(EINVFN);

	return(ixlseek(f,n));
}

/*
	Error returns
		ERANGE
		EINTRN

	Last modified	LTG	31 Jul 85

	NOTE:	This function returns ERANGE and EINTRN errors, which are new error
		numbers I just made up (that is, they were not defined by the BIOS
		or by PC DOS).
*/

long
ixlseek(p,n)
OFD *p;
long n;
{
	int clnum,clx,curnum,i,r;
	DMD *dm;

	if (n > p->o_fileln)
		return(ERANGE);

	if (n < 0)
		return(ERANGE);

	dm = p->o_dmd;
	if (!n)
	{
		clx = 0;
		p->o_curbyt = 0;
		goto fillin;
	}

	/* do we need to start from the beginning ? */

	clnum = divmod(&p->o_curbyt,n,dm->m_clblog);

	if (p->o_curcl && (n >= p->o_bytnum))
	{
		curnum = p->o_bytnum >> dm->m_clblog;
		clnum -= curnum;
		clnum += ((!p->o_curbyt)||(p->o_curbyt == dm->m_clsizb)) && p->o_bytnum;
		clx = p->o_curcl;
	}
	else
		clx = p->o_strtcl;

	for (i=1; i < clnum; i++)
		if ((clx = getcl(clx,dm)) == -1)
			return(-1);

	/* go one more except on cluster boundary */

	if (p->o_curbyt && clnum)
		clx = getcl(clx,dm);

fillin:	p->o_curcl = clx;
	p->o_currec = cl2rec(clx,dm);
	p->o_bytnum = n;

	return(n);
}

int
divmod(modp,divdnd,divsor)
int *modp,divsor;	/* divsor is log2 of actual divisor */
long divdnd;
{
	*modp = divdnd & logmsk[divsor];

	return(divdnd >> divsor);
}
                                                                                